! This program computes trajectories and saves them. One line per time step.
program ham
  use mpi

  implicit none

  integer ierr,myid,Nprocs,Nxstart,Nxend,Nystart,Nyend,Nzstart,Nzend

  double precision angle, Bt, Tilt, SWVel, SWDen, msw, ALindex, gLAT, gMLT, &
       maximum_step_length, saving_step_length
  logical UseAL, dumpfields, usedump, use_gyro_period, use_max_step_length 

  integer iyear, iday, Kp, isec, iy, Npart, Nsteps, ii, jj, sjj, kk, &
       number_of_steps, initialiser, Nretries, attempts
  
  character scdatafilename*200, Parallel*1

  real step_since_saving(3), time_spent
  real parmod(10), Xgse, Ygse, Zgse, pdyn, Dst, ByIMF, BzIMF, &
       Tsygvxgse, Tsygvygse, Tsygvzgse, &
       G1, G2, W1, W2, W3, W4, W5, W6, &
       vxgse, vygse, vzgse, &
       dhtime, hour, minute, Bgsw(3), Bgse(3), posgse(3), velgse(3)

  real test

  real Lside, vth

  double precision dt, dt_1, dt_2, dt0, dt1, dt2, mass, charge, &
       forward, &
       altobs, rEarth,  rr, cyclotronfreq, pi
  double precision xmin, xmax, ymin, ymax, zmin, zmax
  integer Nx, Ny, Nz

  double precision Lx, Ly, Lz, dxyz(3), E0(3), Eperp(3)
  double precision, allocatable :: U(:,:,:), E(:,:,:,:)

  real, allocatable :: position_init_gse(:,:), &
       position_init_gsw(:,:), velocity_init_gse(:,:), velocity_init_gsw(:,:)
	   
  double precision, allocatable :: position(:,:), saving_position(:,:), &
       saving_time(:,:), velocity(:,:), vv(:), mu(:), pangle(:), &
       saving_B(:,:), saving_E(:,:), saving_FL(:,:)

  real field_line(40000,3), xf, yf, zf
  integer field_line_points


  character(8)  :: date
  character(10) :: time
  character(5)  :: zone
  integer,dimension(8) :: values

#ifdef _T01_
  external T01_01, IGRF_GSW_08
#elif _TS05_
  external T04_s, IGRF_GSW_08
#elif _T96_
  external T96_01, IGRF_GSW_08
#endif

! Initialise a couple of things

  parameter(rEarth = 6371.2d3) ! Must be the same as Tsyganenko's Earth radius
  parameter(pi = 3.14159265358979323846264338327950288419716d0)
  parameter(Nretries = 10)

! Creates the string for the directory where we load/save the files
#define STRINGIFY_(x) x
#define STRINGIFY(x) STRINGIFY_(x)

! Get the processes info
CALL MPI_INIT(ierr)
CALL MPI_COMM_RANK(MPI_COMM_WORLD, myid, ierr)
CALL MPI_COMM_SIZE(MPI_COMM_WORLD, Nprocs, ierr)

!Time to compute the program
if ( myid == 0) then
! using keyword arguments
call date_and_time(date,time,zone,values)
call date_and_time(DATE=date,ZONE=zone)
call date_and_time(TIME=time)
call date_and_time(VALUES=values)
print '(a,2x,a,2x,a)', date, time, zone
end if

! Tell the user what the field models are
#ifdef _T01_
  write (*,*) 'Running the ham test particle code with the T01 and IGRF models'
#elif _TS05_
  write (*,*) 'Running the ham test particle code with the TS05 (aka TS04s) &
       and IGRF models'
#elif _T96_
  write (*,*) 'Running the ham test particle code with the T96 and IGRF models'
#endif
end if

  
! Read input data
  attempts = 0
  call GetInput(iyear, iday, dhtime, pdyn, Dst, ByIMF, BzIMF,&
       Tsygvxgse, Tsygvygse, Tsygvzgse, &
       G1, G2, W1, W2, W3, W4, W5, W6, &
       Xgse, Ygse, Zgse, &
       vxgse, vygse, vzgse, Kp, Npart, Nsteps, vth, Lside, dt0, mass, charge, &
       forward, &
       altobs, initialiser, scdatafilename, Nx, Ny, Nz, Parallel, &
       xmin, xmax, ymin, ymax, zmin, zmax, &
       msw, ALindex, UseAL, dumpfields, usedump, &
       number_of_steps, maximum_step_length, use_gyro_period, &
       use_max_step_length, saving_step_length,attempts,Nretries)

  dt_1 = dt0
  dt_2 = dt0
  dt   = dt0
  
! Allocate variables for initial position and velocity, and for 
! particle position and velocity during flight.
  allocate(position_init_gse(Npart,3))
  allocate(position_init_gsw(Npart,3))
  allocate(velocity_init_gse(Npart,3))
  allocate(velocity_init_gsw(Npart,3))
  allocate(position(Nsteps,3))
  allocate(saving_position(Nsteps,3))
  allocate(saving_time(Nsteps,1))
  allocate(saving_B(Nsteps,3))
  allocate(saving_E(Nsteps,3))
  allocate(saving_FL(Nsteps,9))
  allocate(velocity(Nsteps,3))
  allocate(vv(Nsteps))
  allocate(mu(Nsteps))
  allocate(pangle(Nsteps))
  allocate(U(Nx,Ny,Nz)) ! Potential
  allocate(E(3,Nx,Ny,Nz)) ! Electric field

! Initialise
  E = 0.0d0
  U = 0.0d0

! This is something Tsyganenko wants us to do before calling GSWGSE_08 for 
! the first time (and subsequently every time the values of iyear, iday,
! ihour, min, and isec have been changed, and the solar wind velocity too,
! I suppose).
  iy=iyear
  hour=aint(dhtime)
  minute=aint((dhtime-hour)*60.0);
  isec=int(((dhtime-hour)*60.0-minute)*60.0)
  call recalc_08(iyear,iday,int(hour),int(minute),isec, &
       Tsygvxgse,Tsygvygse,Tsygvzgse)


  parmod=0.0 ! initialise all elements first.
  parmod(1) = pdyn  ! Solar wind dynamic pressure [nPa]
  parmod(2) = Dst   ! Dst index [nT]
  parmod(3) = ByIMF ! IMF By component [nT]
  parmod(4) = BzIMF ! IMF Bz component [nT]
#ifdef _T01_
  ! In T01 the first four elements of parmod contain solar wind
  ! parameters used as input to the magnetic field model. If Kp is
  ! set in the input file, that is just a dummy variable. 
  parmod(5) = G1
  parmod(6) = G2
#elif _TS05_
  ! In TS05 the first four elements of parmod contain solar wind
  ! parameters used as input to the magnetic field model. If Kp is
  ! set in the input file, that is just a dummy variable. 
  parmod(5) = W1
  parmod(6) = W2
  parmod(7) = W3
  parmod(8) = W4
  parmod(9) = W5
  parmod(10)= W6
#elif _T96_
  ! In T96 the first four elements of parmod contain solar wind
  ! parameters used as input to the magnetic field model. If Kp is
  ! set in the input file, that is just a dummy variable. 
#endif
  
! System size
  Lx = xmax - xmin
  Ly = ymax - ymin
  Lz = zmax - zmin

! Set grid cell size
  dxyz(1) = Lx/(Nx-1)
  dxyz(2) = Ly/(Ny-1)
  dxyz(3) = Lz/(Nz-1)

  if ( usedump ) then
     attempts = 0
     call LoadDump(U, E, Nx, Ny, Nz, attempts, Nretries)
  else

! Find potentials for the potential matrix using Weimer's model 
    ! Split the domain for multi-proc
     call DivideRegion(Nx, Ny, Nz, Parallel, Nprocs, myid, &
        Nxstart, Nxend, Nystart, Nyend, Nzstart, Nzend)

     call FindPotentials(Nx, Ny, Nz, Nxstart, Nxend, &
         Nystart, Nyend, Nzstart, Nzend, &
         xmin, ymin, zmin, dxyz, Kp, parmod, & 
         ByIMF, BzIMF, Tsygvxgse, Tsygvygse, &
         Tsygvzgse, pdyn, msw, ALindex, UseAL, U)

     ! Send data to Proc 0
      call SendReceiveData(Nx, Ny, Nz, Nxstart, Nxend, Nystart, Nyend, &
         Nzstart, Nzend, Parallel, myid, Nprocs, ierr, U)

      if (myid == 0) then
       ! Compute the electric field
        call NegGradientOpen(U, E, Nx, Ny, Nz, dxyz)


! Write E and U if so desired

        if (dumpfields) then
          attempts = 0
          call DumpScalarField(U,Nx,Ny,Nz,'phi_',0, attempts, Nretries)
          attempts = 0
          call DumpVectorField(E,Nx,Ny,Nz,'Efi_',0, attempts, Nretries)
          attempts = 0
          call DumpDump(U, E, Nx, Ny, Nz, attempts, Nretries)
        end if
      end if      ! ends run on proc 0


  end if

if (myid == 0) then

! Give us particles! 
! This is done differently depending on the initialiser that is chosen 
! in the file inputham.m.

select case (initialiser)
case(1)
   call InitialiseParticles_01(velocity_init_gse,position_init_gse, &
        Xgse,Ygse,Zgse,vxgse,vygse,vzgse,vth,Lside,Npart)
   ! Convert our input GSE coordinates to GSW. The strategy is to perform 
   ! the calculations in GSW and to convert the results just before 
   ! saving them. 
   do ii = 1, Npart
      call gswgse_08(position_init_gsw(ii,1), &
           position_init_gsw(ii,2),position_init_gsw(ii,3), &
           position_init_gse(ii,1), &
           position_init_gse(ii,2),position_init_gse(ii,3),-1)
      call gswgse_08(velocity_init_gsw(ii,1), &
           velocity_init_gsw(ii,2),velocity_init_gsw(ii,3), &
           velocity_init_gse(ii,1), &
           velocity_init_gse(ii,2),velocity_init_gse(ii,3),-1)
   end do

case(2)
   attempts = 0
   call InitialiseParticles_02(velocity_init_gsw, position_init_gsw, &
        Npart, mass, E, scdatafilename, Nx, Ny, Nz, &
        xmin, ymin, zmin, dxyz, Kp, parmod, attempts, Nretries)
   ! The output of InitialiseParticles_02 is already in GSW coordinates; 
   ! therefore we do not convert anything here.

case(3)
   attempts = 0
   call InitialiseParticles_03(velocity_init_gsw, position_init_gsw, &
        Npart, mass, E, scdatafilename, Nx, Ny, Nz, &
        xmin, ymin, zmin, dxyz, Kp, parmod, attempts, Nretries)
   ! The output of InitialiseParticles_03 is already in GSW coordinates; 
   ! therefore we do not convert anything here.


case default
   write (*,*) 'initialiser =', initialiser, ' is illegal!'
   stop
end select

  
! Open the output files. Any old file by the same name will 
! unceremoniously be exterminated.
  open(unit=2,file='outputtrajectories.dat', &
      status='replace',err=97)
  open(unit=3,file='outputfieldlines.dat', &
      status='replace',err=97)


! Here starts the main loop that computes the trajectories and saves them
  do ii=1,Npart

    !Reinitiate everything for the next particles!
    position=0.0d0
    velocity=0.0d0
    saving_time=0.0d0
    saving_position=0.0d0
    dt_1 = dt0 ! For first time step - added by KiAl
    dt_2 = dt0
    dt   = dt0


! Trace the field lines associated with the initial position of each 
! particles

  CALL TraceFieldLine(real(position_init_gsw(ii,:)/rEarth), &
      0.1, 0.0001, 60.0, (1.0+100.0/6371.2), Kp, parmod, &
      field_line, 40000)
  
! Converts GSW to GSE and write field line into the file
  write(3,*,err=98) '% Field Line ', ii
  do kk = 1, 40000
! I check that the value is not just 0 (useless)
    if (field_line(kk,1) /= 0.0) then
      call GSWGSE_08(field_line(kk,1), field_line(kk,2), &
          field_line(kk,3), field_line(kk,1), field_line(kk,2), &
          field_line(kk,3), 1)
      write(3,*,err=98) field_line(kk,1), field_line(kk,2), &
          field_line(kk,3)
    end if
  end do

! Reset the field line to zero for the next loop
    field_line = 0.0
    xf = 0.0
    yf = 0.0
    zf = 0.0

! Convert the initial position and velocity to double precision
     position(1,:)=dble(position_init_gsw(ii,:))
     velocity(1,:)=dble(velocity_init_gsw(ii,:))


! Find initial B-field
#ifdef _T01_
     call bfield(real(position(1,1)/rEarth),real(position(1,2)/rEarth), &
          real(position(1,3)/rEarth),Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T01_01)
#elif _TS05_
     call bfield(real(position(1,1)/rEarth),real(position(1,2)/rEarth), &
          real(position(1,3)/rEarth),Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T04_s)
#elif _T96_
     call bfield(real(position(1,1)/rEarth),real(position(1,2)/rEarth), &
          real(position(1,3)/rEarth),Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T96_01)
#endif

! Interpolate to find E-field
     call InterpolateVectorField(E, Nx, Ny, Nz, dxyz, xmin, ymin, zmin, &
          position(1,1), position(1,2), position(1,3), E0)

! Compute perpendicular E, i.e. remove artificial parallel component
     call PerpendicularEfield(E0, Bgsw, Eperp)

! Take half a step back to be ready for leapfrog action
     call Boris(velocity(1,:),velocity(1,:),Eperp,Bgsw*1e-9,-forward*dt0/2.0d0, &
          mass,charge)

!trajectory saving index reset for next particle
    sjj = 1
    step_since_saving(1) = 0.0
    step_since_saving(2) = 0.0
    step_since_saving(3) = 0.0
    time_spent = 0.0

! Compute the trajectory - start of inner loop
     do jj=1,Nsteps-1
! Find B-field
#ifdef _T01_
        call bfield(real(position(jj,1)/rEarth), &
             real(position(jj,2)/rEarth),real(position(jj,3)/rEarth), &
             Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T01_01)
#elif _TS05_
        call bfield(real(position(jj,1)/rEarth), &
             real(position(jj,2)/rEarth),real(position(jj,3)/rEarth), &
             Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T04_s)
#elif _T96_
        call bfield(real(position(jj,1)/rEarth), &
             real(position(jj,2)/rEarth),real(position(jj,3)/rEarth), &
             Bgsw(1),Bgsw(2),Bgsw(3),Kp,parmod,T96_01)
#endif

! Interpolate to find E-field
        call InterpolateVectorField(E, Nx, Ny, Nz, dxyz, xmin, ymin, zmin, &
             position(jj,1), position(jj,2), position(jj,3), E0)

! Compute perpendicular E, i.e. remove artificial parallel component
        call PerpendicularEfield(E0, Bgsw, Eperp)

! Calculate timestep
! first based on cyclotron frequency omega0 = (q/m)B
        if (use_gyro_period) then
           cyclotronfreq = (charge/mass) * &
                sqrt(dble(Bgsw(1)**2.0 + Bgsw(2)**2.0 + Bgsw(3)**2.0))*1.0d-9
           dt1 = abs(2.0d0*pi / dble(number_of_steps) / cyclotronfreq)
        else 
           dt1 = 1.0d3
        end if

! based on maximum_step_length
        if (use_max_step_length) then
           dt2 = maximum_step_length / sqrt(velocity(jj,1)**2.0d0 + &
                velocity(jj,2)**2.0d0 + velocity(jj,3)**2.0)
        else
           dt2=1.0d3
        end if

! choose and calculate mean value
        dt_1 = min(dt0, dt1, dt2)   ! New timestep
        dt = 0.5d0*(dt_1 + dt_2)    ! Mean of new and old (used for velocity)
        dt_2 = dt_1                 ! Update old for the next time

! Boris gives us the new velocity
        call Boris(velocity(jj,:),velocity(jj+1,:),Eperp,Bgsw*1e-9,forward*dt, &
             mass,charge)

!--------------------------------------------------
!   Save vv, alpha
!--------------------------------------------------

! Now, push!
        position(jj+1,:) = position(jj,:) + velocity(jj+1,:)*forward*dt_1

! Calculate step length
        step_since_saving(1) = step_since_saving(1) + position(jj+1,1) &
             - position(jj,1)
        step_since_saving(2) = step_since_saving(2) + position(jj+1,2) &
             - position(jj,2)
        step_since_saving(3) = step_since_saving(3) + position(jj+1,3) &
             - position(jj,3)
        time_spent = time_spent + forward*dt_1

! Saving the very first time step and if the step is long
        if ( jj == 1 .OR. &
           sqrt(step_since_saving(1)**2.0 + step_since_saving(2)**2.0 + &
           step_since_saving(3)**2.0) .GE. saving_step_length) then

           saving_position(sjj,:) = position(jj,:)
           saving_time(sjj,1) = time_spent

! Compute vv, mu and pangle, i.e. the speed magnetic moment and 
! pitch angle, so that these can be saved later. 
! This approximation uses the B-field at time step jj and the 
! velocity at jj+1, which is half a time step later. However, 
! as long as there is no E-field the result is exactly the same.
           call SpeedMomentAngle(vv(sjj),mu(sjj),pangle(sjj), &
               velocity(jj+1,:),Bgsw,mass)

           sjj = sjj + 1
           step_since_saving(1) = 0.0
           step_since_saving(2) = 0.0
           step_since_saving(3) = 0.0
           field_line = 0.0
        end if

! Check that we are still there
        rr=sqrt(position(jj+1,1)**2+position(jj+1,2)**2+position(jj+1,3)**2)
        if (rr < rEarth + altobs .or. &
             position(jj+1,1)<xmin .or. position(jj+1,1)>xmax .or. &
             position(jj+1,2)<ymin .or. position(jj+1,2)>ymax .or. &
             position(jj+1,3)<zmin .or. position(jj+1,3)>zmax ) then
           exit
        end if

     end do

! Save the trajectory
     write (2,*,err=98) '% trajectory ', ii
     do kk=1,sjj-1
        
! Convert output to GSE coordinates
        call gswgse_08(real(saving_position(kk,1)), &
             real(saving_position(kk,2)), real(saving_position(kk,3)), &
             posgse(1),posgse(2),posgse(3),1)

! Save positions, time, speed, magnetic moment and pitch angle
        write (2,fmt='(7E14.6)',err=98) &
             posgse(1), posgse(2), posgse(3), saving_time(kk,1), &
             vv(kk), mu(kk), pangle(kk)
     end do
  end do
  
! Close the output file
  close(2,err=99)
  close(3,err=99)

if ( myid == 0) then
! using keyword arguments
call date_and_time(date,time,zone,values)
call date_and_time(DATE=date,ZONE=zone)
call date_and_time(TIME=time)
call date_and_time(VALUES=values)
print '(a,2x,a,2x,a)', date, time, zone
end if

  ! Finish process 0
  CALL MPI_FINALIZE(ierr)

  stop ! This ends normal execution of the main program
  
! Error handling section
97 write (*,*) 'ham: error in open statement'
  goto 100
98 write (*,*) 'ham: error in write statement'
  close(1,err=99)
  goto 100
99 write (*,*) 'ham: error in close statement'
100 write (*,*) 'exiting'
  
end if           ! ends if (myid==0)

! Finish all processes but 0
CALL MPI_FINALIZE(ierr)

end program ham

